/*
 * Copyright 2019, Cypress Semiconductor Corporation or a subsidiary of
 * Cypress Semiconductor Corporation. All Rights Reserved.
 *
 * This software, including source code, documentation and related
 * materials ("Software"), is owned by Cypress Semiconductor Corporation
 * or one of its subsidiaries ("Cypress") and is protected by and subject to
 * worldwide patent protection (United States and foreign),
 * United States copyright laws and international treaty provisions.
 * Therefore, you may use this Software only as provided in the license
 * agreement accompanying the software package from which you
 * obtained this Software ("EULA").
 * If no EULA applies, Cypress hereby grants you a personal, non-exclusive,
 * non-transferable license to copy, modify, and compile the Software
 * source code solely for use in connection with Cypress's
 * integrated circuit products. Any reproduction, modification, translation,
 * compilation, or representation of this Software except as specified
 * above is prohibited without the express written permission of Cypress.
 *
 * Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. Cypress
 * reserves the right to make changes to the Software without notice. Cypress
 * does not assume any liability arising out of the application or use of the
 * Software or any product or circuit described in the Software. Cypress does
 * not authorize its products for use in any products where a malfunction or
 * failure of the Cypress product may reasonably be expected to result in
 * significant property damage, injury or death ("High Risk Product"). By
 * including Cypress's product in a High Risk Product, the manufacturer
 * of such system or application assumes all risk of such use and in doing
 * so agrees to indemnify Cypress against all liability. 
 */

/** @file
 *
 * Mesh
 *
 * A mesh network uses LE advertisements to send messages between node.
 * The original data packet could be generated by a node itself, like Can-You-
 * See-Me packet, or could be received from outside via LE or wired interface.
 * The node sends the data packet using LE advertisement. When a node received
 * a message, if the message is for itself only, it performs the action specified
 * in the message; otherwise it rebroadcasts the same message. Since advertisement
 * packet delivery is not guaranteed, each message is broadcasted multiple times
 * - the repeat time and interval can be configured. Each node also makes sure
 * one message will only be handled once and only once at each node.
 *
 * This is a mesh sample application. This node implements a proxy interface
 * for a LE-capable device to connect with it. the device sends the
 * mesh command packets to this node using a LE connection. This node sends the
 * command packet out on behalf of the device - hence this node performs
 * proxy function for the device to the mesh network. The interface between
 * a proxy node and the device is up to the customers - the one here is
 * just an example.
 */

#include "sparcommon.h"
#include "bt_types.h"
#include "wiced_bt_ble.h"
#include "wiced_bt_cfg.h"
#include "wiced_hal_gpio.h"
#include "wiced_bt_uuid.h"
#include "wiced_hal_nvram.h"
#include "wiced_platform.h"
#include "wiced_bt_mesh_model_defs.h"
#include "wiced_bt_mesh_models.h"
#include "wiced_bt_mesh_event.h"
#include "wiced_bt_mesh_provision.h"
#include "wiced_bt_mesh_app.h"
#include "wiced_bt_mesh_client.h"
#include "wiced_bt_mesh_core.h"
#include "wiced_bt_trace.h"
#include "mesh_application.h"
#include "hci_control_api.h"
#include "wiced_memory.h"
#if ( defined(CYW20735B0) || defined(CYW20735B1) )
#include "wiced_gki.h"
#endif
#include "wiced_bt_ota_firmware_upgrade.h"
#include "wiced_hal_puart.h"
#ifdef CYW20706A2
#include "wiced_bt_hci_defs.h"
#else
#include "hcidefs.h"
#endif
#include "wiced_hal_wdog.h"
#include "wiced_bt_stack.h"
#include "wiced_hal_rand.h"
#include "wiced_bt_l2c.h"
#include "wiced_bt_dev.h"
#if ( defined(CYW20706A2) || defined(CYW20719B1) || defined(CYW20719B0) || defined(CYW20721B1) || defined(CYW20735B0) || defined(CYW43012C0) )
#include "wiced_bt_app_common.h"
#endif
//#define _DEB_DELAY_START_SEC 10
#ifdef _DEB_DELAY_START_SEC
#include "wiced_timer.h"
#endif

// Comment out next line if we don't want to support GATT provisioning
#define MESH_SUPPORT_PB_GATT

// If defined then all NVRAM operations are executed on MCU via WICED-HCI
//#define MESH_APPLICATION_MCU_MEMORY

#ifdef MESH_APPLICATION_MCU_MEMORY
uint16_t mesh_application_mcu_memory_read(uint16_t id, uint16_t buf_len, uint8_t *p_buf);
uint16_t mesh_application_mcu_memory_write(uint16_t id, uint16_t data_len, const uint8_t *p_data);
wiced_result_t mesh_application_process_hci_cmd(uint16_t opcode, const uint8_t *p_data, uint16_t data_len);
wiced_result_t mesh_application_send_hci_event(uint16_t opcode, const uint8_t *p_data, uint16_t data_len);
#endif


/******************************************************
 *          Constants
 ******************************************************/
// The power table is 3, -1, -5, -9
#define TX_DBM   4


/******************************************************
 *          Structures
 ******************************************************/

extern wiced_bt_mesh_core_config_t          mesh_config;

#pragma pack(1)
#ifndef PACKED
#define PACKED
#endif

// represents data saved in the NVRAM with index mesh_nvm_idx_seq
typedef PACKED struct
{
    uint32_t    seq;        // self SEQ
    uint16_t    size;       // Size of the RPL - number of RPL items
} mesh_app_rpl_init_t;

#define MESH_APP_RPL_ITEM_PREV_IVI_FLAG 0x8000
// Represents one RPL entry in the NVRAM
typedef PACKED struct
{
    uint16_t    addr;       // Bit MESH_APP_RPL_ITEM_PREV_IVI_FLAG means previous IVI
    uint8_t     seq[3];     // latest SEQ for SRC addr
} mesh_app_rpl_item_t;

#pragma pack()

// current value of own SEQ and RPL size
static mesh_app_rpl_init_t mesh_app_rpl_init = { 0 };
#ifdef CYW20706A2
// On 20706 we keep all RPL entries in the one NVRAM index
static mesh_app_rpl_item_t *mesh_app_rpl_items = NULL;
#endif

// Application can set this handler to process if it implements models layer (including configuration).
wiced_bt_mesh_core_received_msg_handler_t p_app_model_message_handler = NULL;

static wiced_bt_mesh_core_received_msg_handler_t get_msg_handler_callback(uint16_t company_id, uint16_t opcode, uint16_t *p_model_id, wiced_bool_t *p_dont_save_rpl);
static wiced_bool_t mesh_publication_callback(uint8_t elem_idx, uint16_t company_id, uint16_t model_id, uint16_t period);

wiced_bt_mesh_core_received_msg_handler_t p_proxy_status_message_handler = NULL;

/******************************************************
 *          Function Prototypes
 ******************************************************/

// Local functions
       wiced_result_t           mesh_management_cback(wiced_bt_management_evt_t event, wiced_bt_management_evt_data_t *p_event_data);
       void                     mesh_application_init(void);
static void                     mesh_interrupt_handler(void* user_data, uint8_t value);
static uint32_t                 mesh_nvram_access(wiced_bool_t write, int inx, uint8_t* node_info, uint16_t len, wiced_result_t *p_result);
static uint8_t                  mesh_fault_test(uint8_t element, uint8_t test_id, uint16_t company_id, uint8_t fault_array_size, uint8_t *fault_array);
static void                     mesh_state_changed_cb(wiced_bt_mesh_core_state_type_t type, wiced_bt_mesh_core_state_t *p_state);
static void                     mesh_adv_report(wiced_bt_ble_scan_results_t *p_scan_result, uint8_t *p_adv_data);
static uint8_t                  number_of_elements_with_model(uint16_t model_id);
static void                     mesh_setup_nvram_ids(void);
static void                     mesh_start_stop_scan_callback(wiced_bool_t start, wiced_bool_t is_active);

static wiced_bool_t             mesh_application_seq_save(wiced_bt_mesh_core_state_seq_t *p_seq);
static wiced_bool_t             mesh_application_seq_init(void);
static wiced_bool_t             mesh_application_rpl_clr(void);

#ifdef MESH_HOMEKIT_COMBO_APP
extern void mesh_provisioning_state_changed(wiced_bool_t provisioned);
extern void homekit_factory_reset();
#endif

extern wiced_bt_cfg_settings_t wiced_bt_cfg_settings;
extern const wiced_bt_cfg_buf_pool_t wiced_bt_cfg_buf_pools[];
extern uint16_t wiced_bt_mesh_core_lpn_get_friend_addr(void);
extern void wiced_bt_mesh_core_shutdown(void);

// NVM index for SEQ
uint16_t mesh_nvm_idx_seq;

/******************************************************
 *          Variables Definitions
 ******************************************************/
// Use hardcoded PTS default priv key. In real app it will be generated once and written into OTP memory
uint8_t pb_priv_key[WICED_BT_MESH_PROVISION_PRIV_KEY_LEN] = { 0x52, 0x9A, 0xA0, 0x67, 0x0D, 0x72, 0xCD, 0x64, 0x97, 0x50, 0x2E, 0xD4, 0x73, 0x50, 0x2B, 0x03, 0x7E, 0x88, 0x03, 0xB5, 0xC6, 0x08, 0x29, 0xA5, 0xA3, 0xCA, 0xA2, 0x19, 0x50, 0x55, 0x30, 0xBA };

//WICED_TRUE is we are provisioner
wiced_bool_t mesh_config_client = WICED_FALSE;
wiced_bool_t node_authenticated = WICED_FALSE;
wiced_bool_t pb_gatt_in_progress = WICED_FALSE;

/******************************************************
 *               Function Definitions
 ******************************************************/

/*
*  Entry point to the application. Set device configuration and start BT
*  stack initialization.  The actual application initialization will happen
*  when stack reports that BT device is ready.
*/
#ifndef MESH_HOMEKIT_COMBO_APP
#if (defined(CYW20719B0) || defined(CYW20719B1) || defined(CYW20721B1) ||  defined(CYW20706A2))
APPLICATION_START()
#else
void application_start(void)
#endif
#else // MESH_HOMEKIT_COMBO_APP
void mesh_application_start()
#endif
{
    mesh_app_hci_init();

#ifndef WICEDX_LINUX
    // If application wants to control the hardware, call appropriate initialization function.  
    // Otherwise use default processing of the mesh application library.
    if (wiced_bt_mesh_app_func_table.p_mesh_app_hw_init != NULL)
    {
        wiced_bt_mesh_app_func_table.p_mesh_app_hw_init();
    }
    else
    {
        /* Configure buttons available on the platform */
#if defined(CYW20706A2)
        wiced_hal_gpio_configure_pin(WICED_GPIO_BUTTON, WICED_GPIO_BUTTON_SETTINGS(GPIO_EN_INT_BOTH_EDGE), WICED_GPIO_BUTTON_DEFAULT_STATE);
        wiced_hal_gpio_register_pin_for_interrupt(WICED_GPIO_BUTTON, mesh_interrupt_handler, NULL);
#elif (defined(CYW20735B0) || defined(CYW20719B0) || defined(CYW20721B0))
        wiced_hal_gpio_register_pin_for_interrupt(WICED_GPIO_PIN_BUTTON, mesh_interrupt_handler, NULL);
        wiced_hal_gpio_configure_pin(WICED_GPIO_PIN_BUTTON, WICED_GPIO_BUTTON_SETTINGS, GPIO_PIN_OUTPUT_LOW);
#else
        wiced_platform_register_button_callback(WICED_PLATFORM_BUTTON_1, mesh_interrupt_handler, NULL, GPIO_EN_INT_BOTH_EDGE);
#endif
    }
#ifndef MESH_HOMEKIT_COMBO_APP
    // Register call back and configuration with stack
    wiced_bt_stack_init(mesh_management_cback, &wiced_bt_cfg_settings, wiced_bt_cfg_buf_pools);
#endif
#endif
    // Currently we can support up to 4 network keys.
    wiced_bt_mesh_core_net_key_max_num = 4;
    wiced_bt_mesh_core_app_key_max_num = 8;
    wiced_bt_mesh_scene_max_num            = 10;
    wiced_bt_mesh_scheduler_events_max_num = 16; // PTS test uses index 15 (MMDL/SR/SCHS/BV-01-C )
    
    // setup NVRAM IDs which will be used by core and models
    mesh_setup_nvram_ids();

    WICED_BT_TRACE("Mesh Start\n");
}

#ifdef _DEB_DELAY_START_SEC
wiced_timer_t    mesh_delay_start_timer;
void mesh_delay_start_timer_callback(TIMER_PARAM_TYPE arg)
{
#ifdef MESH_APPLICATION_MCU_MEMORY
    mesh_application_send_hci_event(HCI_CONTROL_EVENT_DEVICE_STARTED, NULL, 0);
#else
    mesh_application_init();
#endif
}

void mesh_delay_start_init(void)
{
    memset(&mesh_delay_start_timer, 0, sizeof(wiced_timer_t));
    wiced_init_timer(&mesh_delay_start_timer, mesh_delay_start_timer_callback, 0, WICED_SECONDS_TIMER);
    wiced_start_timer(&mesh_delay_start_timer, _DEB_DELAY_START_SEC);
}

#endif

/*
* bt/ble link management callback
*/
wiced_result_t mesh_management_cback(wiced_bt_management_evt_t event, wiced_bt_management_evt_data_t *p_event_data)
{
    wiced_bt_ble_advert_mode_t          *p_mode;
    wiced_result_t                      result = WICED_BT_SUCCESS;

    // WICED_BT_TRACE("mesh_management_cback: %x\n", event);
    //test_aes();

    switch (event)
    {
        /* Bluetooth  stack enabled */
    case BTM_ENABLED_EVT:
#ifdef _DEB_DELAY_START_SEC
        mesh_delay_start_init();
#else
#ifdef MESH_APPLICATION_MCU_MEMORY
        mesh_application_send_hci_event(HCI_CONTROL_EVENT_DEVICE_STARTED, NULL, 0);
#else
        mesh_application_init();
#endif
#endif
        break;

    case BTM_DISABLED_EVT:
        break;

    case BTM_BLE_ADVERT_STATE_CHANGED_EVT:
        p_mode = &p_event_data->ble_advert_state_changed;
        WICED_BT_TRACE("Advertisement State Changed:%d\n", *p_mode);
        if (*p_mode == BTM_BLE_ADVERT_OFF)
        {
            WICED_BT_TRACE("adv stopped\n");
            // On failed attempt to connect FW stops all connectable adverts. 20719B1 also receives that event in case of successfull connection
            // If we disconnected then notify core to restart them
            if (!mesh_app_gatt_is_connected())
            {
                wiced_bt_mesh_core_connection_status(0, WICED_FALSE, 0, 20);
            }
        }
        break;

    case BTM_BLE_SCAN_STATE_CHANGED_EVT:
        WICED_BT_TRACE("Scan State Change:%d\n", p_event_data->ble_scan_state_changed);
        break;

    case  BTM_PAIRED_DEVICE_LINK_KEYS_REQUEST_EVT:
        result = WICED_BT_ERROR;
        break;

    default:
        break;
    }

    return result;
}

/**
* Generates random number.
*
* Parameters:
*   random:     Buffer for generated random.
*   len:        Length of the random number to generate.
*
* Return: None
*
*/
void mesh_generate_random(uint8_t* random, uint8_t len)
{
    uint32_t r;
    uint8_t l;
    while (len)
    {
        r = wiced_hal_rand_gen_num();
        l = len > 4 ? 4 : len;
        memcpy(random, &r, l);
        len -= l;
        random += l;
    }
}

#ifndef PTS
extern void utilslib_delayUs(UINT32 delay);
static void mesh_application_gen_bda(uint8_t *non_provisioned_bda, uint8_t *provisioned_bda)
{
    uint32_t            r1[2];
    int                 i;
#ifndef WICEDX_LINUX
    // TRNG might not work while chip is warming up.  Wait 2 seconds.
    // This happens only once during the very first power up, so should not be a problem.
    for (i = 0; i < 2000; i++)
    {
        utilslib_delayUs(1000);
        wiced_hal_wdog_restart();
    }
#endif
    r1[0] = wiced_hal_rand_gen_num();
    r1[1] = wiced_hal_rand_gen_num();
    memcpy(non_provisioned_bda, (uint8_t *)r1, 6);
    /* Valid static random address should have 2 most significant bits set to 1 */
    non_provisioned_bda[0] |= 0xc0;

    r1[0] = wiced_hal_rand_gen_num();
    r1[1] = wiced_hal_rand_gen_num();
    memcpy(provisioned_bda, (uint8_t *)r1, 6);
    /* Valid static random address should have 2 most significant bits set to 1 */
    provisioned_bda[0] |= 0xc0;

    WICED_BT_TRACE("mesh_application_gen_bda: bda:%B provisioned bda:%B\n", non_provisioned_bda, provisioned_bda);
}
#endif

void mesh_application_factory_reset(void)
{
    WICED_BT_TRACE("mesh_application_factory_reset: *************************\n");
    if (wiced_bt_mesh_app_func_table.p_mesh_app_factory_reset)
    {
        wiced_bt_mesh_app_func_table.p_mesh_app_factory_reset();
    }
    wiced_bt_mesh_core_init(NULL);
    node_authenticated = WICED_FALSE;
    pb_gatt_in_progress = WICED_FALSE;
}

void mesh_ota_firmware_upgrade_status_callback(uint8_t status)
{
    WICED_BT_TRACE("mesh_ota_firmware_upgrade_status_callback: status:%d\n", status);
#ifdef CYW20706A2
    // Due to a FW bug in 20706A2, the chip may crash if too many advertisements arrive
    // while we are reading the SFLASH.  To work around the problem, disable for the duration
    // while we are reading.
    if (status == OTA_FW_UPGRADE_STATUS_VERIFICATION_START)
        mesh_start_stop_scan_callback(WICED_FALSE, WICED_FALSE);
    else if (status == OTA_FW_UPGRADE_STATUS_ABORTED)
        mesh_start_stop_scan_callback(WICED_TRUE, WICED_FALSE);
#endif
}

// This function is executed in the BTM_ENABLED_EVT management callback.
void mesh_application_init(void)
{
    wiced_result_t            result;
    uint8_t                   buffer[(6 * 2) + 16];
    wiced_bt_mesh_core_init_t init = { 0 };

    WICED_BT_TRACE("## mesh_application_init free_bytes:%d ##\n", wiced_memory_get_free_bytes());

#ifdef WICEDX_LINUX
    extern void wiced_bt_mesh_core_set_trace_level(uint32_t fids_mask, uint8_t level);
    wiced_bt_mesh_core_set_trace_level(0xffffffff, 4);      //(ALL, TRACE_DEBUG)
    wiced_bt_mesh_core_set_trace_level((1 << 3), 0);        // 3 - FID_MESH_APP__CORE_AES_CCM_C
#endif

#ifndef MESH_HOMEKIT_COMBO_APP
    /* Initialize wiced app */
#if (!defined(CYW20735B1) && !defined(CYW20819A1))
    wiced_bt_app_init();
#endif

#endif

#ifndef WICEDX_LINUX
#ifndef MESH_HOMEKIT_COMBO_APP
    mesh_app_gatt_init();
    wiced_bt_set_pairable_mode(WICED_FALSE, WICED_FALSE);
#endif // MESH_HOMEKIT_COMBO_APP
#endif
    //remember if we are provisioner (config client)
    mesh_config_client = number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_CONFIG_CLNT) > 0 ? WICED_TRUE : WICED_FALSE;

#ifdef PTS
    // initialize core
    // each node shall be assigned a 128-bit UUID known as the Device UUID.
    // Device manufacturers shall follow the standard UUID format and generation
    // procedure to ensure the uniqueness of each Device UUID.
    // for now device_uuid is local bda with hardcoded(0x0f) remaining bytes
    wiced_bt_dev_read_local_addr(init.non_provisioned_bda);
    memcpy(init.provisioned_bda, init.non_provisioned_bda, sizeof(wiced_bt_device_address_t));
    memcpy(&init.device_uuid[0], init.non_provisioned_bda, 6);
    memset(&init.device_uuid[6], 0x0f, 16 - 6);
    // in PTS disable proxy on demand - always advertise proxy service network if when appropriate
    wiced_bt_mesh_core_proxy_on_demand_advert_to = 0;
    WICED_BT_TRACE("UUID:\n");
    WICED_BT_TRACE_ARRAY(init.device_uuid, 16, "");
#else
    if (mesh_nvram_access(WICED_FALSE, NVRAM_ID_LOCAL_UUID, init.device_uuid, 16, &result) != 16)
    {
        // We currently use 2 BDAs, one for the initial scenario. We also keep saved UUID
        mesh_application_gen_bda(init.non_provisioned_bda, init.provisioned_bda);
        memcpy(init.device_uuid, init.non_provisioned_bda, 6);
        memcpy(&init.device_uuid[6], init.provisioned_bda, 6);
        *(uint32_t*)&init.device_uuid[12] = wiced_hal_rand_gen_num();

        mesh_nvram_access(WICED_TRUE, NVRAM_ID_LOCAL_UUID, init.device_uuid, 16, &result);
        WICED_BT_TRACE("result:%x UUID:\n", result);
        WICED_BT_TRACE_ARRAY(init.device_uuid, 16, "");
    }
    else
    {
        memcpy(init.non_provisioned_bda, init.device_uuid, 6);
        memcpy(init.provisioned_bda, &init.device_uuid[6], 6);
    }
#endif

    // Remove this line if MeshClient supports proxy on demand
    wiced_bt_mesh_core_proxy_on_demand_advert_to = 0;

#ifdef MESH_SUPPORT_PB_GATT
    mesh_config.features |= WICED_BT_MESH_CORE_FEATURE_BIT_PB_GATT;
#endif
    init.p_config_data = &mesh_config;
    init.callback = get_msg_handler_callback;
    init.pub_callback = mesh_publication_callback;
    init.proxy_send_callback = mesh_app_proxy_gatt_send_cb;
    init.nvram_access_callback = mesh_nvram_access;
    init.fault_test_cb = mesh_fault_test;
    init.attention_cb = wiced_bt_mesh_app_func_table.p_mesh_app_attention;
    init.state_changed_cb = mesh_state_changed_cb;
    init.scan_callback = mesh_start_stop_scan_callback;
    node_authenticated = WICED_BT_SUCCESS == wiced_bt_mesh_core_init(&init);

#ifndef WICEDX_LINUX
    mesh_app_gatt_db_init(node_authenticated);

    // Initialize own SEQ and RPL
    if (!mesh_application_seq_init())
        mesh_application_factory_reset();

    // Initialize OTA FW upgrade
    if (!wiced_ota_fw_upgrade_init(NULL, mesh_ota_firmware_upgrade_status_callback, mesh_ota_firmware_upgrade_send_data_callback))
    {
        WICED_BT_TRACE("mesh_application_init: wiced_ota_fw_upgrade_init failed\n");
    }
#endif

    wiced_bt_mesh_app_provision_server_init(pb_priv_key, NULL);

    if (wiced_bt_mesh_app_func_table.p_mesh_app_init)
    {
        wiced_bt_mesh_app_func_table.p_mesh_app_init(node_authenticated);
    }
    // Now start mesh picking up tx power set by app in wiced_bt_mesh_core_adv_tx_power
    wiced_bt_mesh_core_start();

    WICED_BT_TRACE("***** Free mem after app_init:%d\n", wiced_memory_get_free_bytes());
}

#ifdef WICEDX_LINUX
// This function is called when application wants to turn off mesh
void mesh_application_deinit(void)
{
    if (wiced_bt_mesh_app_func_table.p_mesh_app_init)
    {
        wiced_bt_mesh_app_func_table.p_mesh_app_init(WICED_FALSE);
    }
    wiced_bt_ble_observe(WICED_FALSE, 0, NULL);
    wiced_bt_mesh_core_shutdown();
}
#endif

// Handle received adv packet
static void mesh_adv_report(wiced_bt_ble_scan_results_t *p_scan_result, uint8_t *p_adv_data)
{
    if (p_scan_result == NULL)
        return;

#if REMOTE_PROVISIONG_SERVER_SUPPORTED
    if (!wiced_bt_mesh_remote_provisioning_adv_packet(p_scan_result, p_adv_data))
#endif
    {
        if (p_scan_result->ble_evt_type == BTM_BLE_EVT_NON_CONNECTABLE_ADVERTISEMENT)
        {
            wiced_bt_mesh_core_adv_packet(p_scan_result->rssi, p_adv_data);
        }
    }
}

// Default interrupt processing.  If application does not want to reset on the application
// button press it shall provide p_mesh_app_hw_init function during initialization and provide its
// own interrupt processing 
void mesh_interrupt_handler(void* user_data, uint8_t pin)
{
#ifndef WICEDX_LINUX
    uint32_t value = wiced_hal_gpio_get_pin_input_status(pin);
    //WICED_BT_TRACE("interrupt_handler: pin:%d value:%d\n", pin, value);

    if (value != 0)
    {
        mesh_application_factory_reset();
#ifdef MESH_HOMEKIT_COMBO_APP
        homekit_factory_reset();
#endif
    }
#endif
}

/*
 * Application provided function to read/write information from/into NVRAM
 */
static uint32_t mesh_nvram_access(wiced_bool_t write, int inx, uint8_t* node_info, uint16_t len, wiced_result_t *p_result)
{
    uint32_t len_res = len;

#ifdef MESH_NVRAM_NOT_SUPPORTED
    *p_result = 0;
    if (write)
        return len;
    else
        return 0;
#else
#ifdef MESH_APPLICATION_MCU_MEMORY
    if (!write)
        len_res = mesh_application_mcu_memory_read(inx, len, node_info);
    else
        len_res = mesh_application_mcu_memory_write(inx, len, node_info);
    *p_result = 0;
#else
    if (!write)
        len_res = wiced_hal_read_nvram(inx, len, node_info, p_result);
    // 0 len means delete
    else if (len != 0)
        len_res = wiced_hal_write_nvram(inx, len, node_info, p_result);
    else
        wiced_hal_delete_nvram(inx, p_result);
#endif
#endif
    //WICED_BT_TRACE("mesh_nvram_access: inx:%x write:%d result:%x len:%d/%d\n", inx, write, *p_result, len, len_res);
    return len_res;
}

/**
* \brief Callback function to invoke a self test procedure of an Element.
* \details Application implements that function to support Health Server Model
* Called by core to get Current Fault.
*
* @param[in]   element             :Index of the elemnt
* @param[in]   test_id             :Identifier of a specific test to be performed
* @param[in]   company_id          :Company ID of the test
* @param[in]   fault_array_size    :Size of the buffer fault_array
* @param[out]  fault_array         :Buffer to receive FaultArray
*
* @return      number of error codes filled into fault_array
*/
static uint8_t mesh_fault_test(uint8_t element, uint8_t test_id, uint16_t company_id, uint8_t fault_array_size, uint8_t *fault_array)
{
    uint8_t     ret = 0;
    WICED_BT_TRACE("mesh_fault_test: element:%d test_id:%d company_id:%x\n", element, test_id, company_id);
    // We support only test_id 0 and company ID from composition data(Cypress). It corresponds to the PTS default valid test ids
    if (test_id != 0 || company_id != mesh_config.company_id)// MESH_COMPANY_ID_CYPRESS)
        return 0xff;
    if (fault_array_size--)
        fault_array[ret++] = 0x01;
    if (fault_array_size--)
        fault_array[ret++] = 0x02;
    if (fault_array_size--)
        fault_array[ret++] = 0x03;
    return ret;
}

/*
 * Returns the status of the Provisioning status of the node.
 * Returns WICED_TRUE: If node is provisioned, WICED_FALSE: If node is not provisioned.
 */
wiced_bool_t mesh_app_node_is_provisioned(void)
{
    return node_authenticated;
}

// Application implements that function to get notified on the periodic publication event
// If period is set to 0, the publication needs to stop. Period between 1 and 0xFFFE indicates
// that Periodic Publication for the specified model is being changed.  Value 0xFFFF indicates
// that the value needs to be published now.
wiced_bool_t mesh_publication_callback(uint8_t elem_idx, uint16_t company_id, uint16_t model_id, uint16_t period)
{
    wiced_bt_mesh_core_received_msg_handler_t   p_message_handler = NULL;
    wiced_bt_mesh_event_t                       *p_event;
    uint8_t                                     idx_model;

    WICED_BT_TRACE("mesh_publication_callback: idx:%d company_id:%x model_id:%x period:%04x\n", elem_idx, company_id, model_id, period);

    if (period != 0xFFFF)
    {
        // Check if application wants to process periodic notifications
        if (wiced_bt_mesh_app_func_table.p_mesh_app_notify_period_set != NULL)
        {
            // core reports, period in 100ms. Convert to milliseconds to send to the app.
            return (wiced_bt_mesh_app_func_table.p_mesh_app_notify_period_set(elem_idx, company_id, model_id, period * 100));
        }
        // Let core handle periodic publications
        return WICED_FALSE;
    }
    if (elem_idx < mesh_config.elements_num)
    {
        // get message handler of specific model
        for (idx_model = 0; idx_model < mesh_config.elements[elem_idx].models_num; idx_model++)
        {
            if (company_id == mesh_config.elements[elem_idx].models[idx_model].company_id
                && model_id == mesh_config.elements[elem_idx].models[idx_model].model_id)
            {
                p_message_handler = (wiced_bt_mesh_core_received_msg_handler_t)mesh_config.elements[elem_idx].models[idx_model].p_message_handler;
                break;
            }
        }
    }
    if (p_message_handler)
    {
        // create event, 0 DST means get info from publication.  Create event will use opcode
        // WICED_BT_MESH_OPCODE_UNKNOWN which will be treated by the library as Get.
        if (NULL != (p_event = wiced_bt_mesh_create_event(elem_idx, company_id, model_id, 0, 0)))
        {
            if (p_message_handler(p_event, NULL, 0))
                return WICED_TRUE;
            wiced_bt_mesh_release_event(p_event);
        }
    }
    else
    {
        WICED_BT_TRACE(" ignored\n");
    }
    return WICED_FALSE;
}

/*
 * Application implements that function to handle received messages. Call each library that this device needs to support.
 */
wiced_bt_mesh_core_received_msg_handler_t get_msg_handler_callback(uint16_t company_id, uint16_t opcode, uint16_t *p_model_id, wiced_bool_t *p_dont_save_rpl)
{
    wiced_bt_mesh_core_received_msg_handler_t p_message_handler = NULL;
    uint8_t                                   idx_elem, idx_model;
    wiced_bt_mesh_event_t                     temp_event;
    uint16_t                                  model_id;

    WICED_BT_TRACE("\ncompany_id:%x opcode:%x\n", company_id, opcode);

    // Each model present on the element contains the callback which will be executed.
    // If message handler returns that callback, that means that Opcode is for that model.
    // The special case is Proxy Status messages, which do not belong to any model.
    if ((company_id == MESH_COMPANY_ID_UNUSED) && (p_proxy_status_message_handler != NULL))
    {
        p_message_handler = p_proxy_status_message_handler;
    }
    else
    {
        temp_event.company_id = company_id;
        temp_event.opcode     = opcode;
        temp_event.model_id   = 0xffff;   // it is a sign of special mode for model to just return true if the opcode is for that model, without message handling.
                                          // Model library changes model_id to any other value if it wants do disable RPL saving for its messages

        for (idx_elem = 0; idx_elem < mesh_config.elements_num; idx_elem++)
        {
            for (idx_model = 0; idx_model < mesh_config.elements[idx_elem].models_num; idx_model++)
            {
                if (company_id != mesh_config.elements[idx_elem].models[idx_model].company_id)
                    continue;
                p_message_handler = (wiced_bt_mesh_core_received_msg_handler_t)mesh_config.elements[idx_elem].models[idx_model].p_message_handler;
                if (p_message_handler == NULL)
                    continue;
                if (!p_message_handler(&temp_event, NULL, 0))
                    continue;
                model_id = mesh_config.elements[idx_elem].models[idx_model].model_id;
                if (p_model_id)
                    *p_model_id = model_id;
                // Check if model wants to disable RPL saving for its messages
                if (temp_event.model_id != 0xffff && p_dont_save_rpl != NULL)
                    *p_dont_save_rpl = WICED_TRUE;
                break;
            }
            if (idx_model < mesh_config.elements[idx_elem].models_num)
                break;
        }
        if (idx_elem >= mesh_config.elements_num)
            p_message_handler = NULL;
    }
    if (p_message_handler == NULL)
    {
        if (p_app_model_message_handler != NULL)
        {
            p_message_handler = p_app_model_message_handler;
        }
        else
        {
            WICED_BT_TRACE("ignored\n");
        }
    }
    return p_message_handler;
}

/*
 * Application implements that function to start/stop scanning as requested by the core
 */
void mesh_start_stop_scan_callback(wiced_bool_t start, wiced_bool_t is_scan_active)
{
    uint16_t loacal_addr = wiced_bt_mesh_core_get_local_addr();
    WICED_BT_TRACE("scan callback: start:%d active:%d loacal_addr:%x pb_gatt:%d\n", start, is_scan_active, loacal_addr, pb_gatt_in_progress);

#ifndef WICEDX_LINUX
    if (start)
    {
        wiced_bt_cfg_settings.ble_scan_cfg.scan_mode = is_scan_active ? BTM_BLE_SCAN_MODE_ACTIVE : BTM_BLE_SCAN_MODE_PASSIVE;
        wiced_bt_cfg_settings.ble_scan_cfg.low_duty_scan_window = 96;
        // If node is LPN in unprovisioned state before PB-ADV start then use low duty scan to save the battery. High dury scan will be restores on PB-ADV start.
        if ((mesh_config.features & WICED_BT_MESH_CORE_FEATURE_BIT_LOW_POWER) != 0 && loacal_addr == 0 && !pb_gatt_in_progress)
            wiced_bt_cfg_settings.ble_scan_cfg.low_duty_scan_window = 48;
    }
#endif
    wiced_bt_ble_observe(start, 0, mesh_adv_report);
}

#ifdef CYW20706A2
// Core initialization initializes the RTC calling wiced_bt_mesh_core_rtc_init() defined in app. It just should call rtc_init().
// For that chip rtc_init should be called from application. Otherwise there is build error at link phase
void wiced_bt_mesh_core_rtc_init(void)
{
    rtc_init();
}
#endif

static void mesh_state_changed_cb(wiced_bt_mesh_core_state_type_t type, wiced_bt_mesh_core_state_t *p_state)
{
    wiced_bt_gatt_status_t  gatt_status;
    wiced_result_t          result;

    switch (type)
    {
    case WICED_BT_MESH_CORE_STATE_TYPE_CLR_RPL:
        // Clear RPL. On error do factory reset
        if(!mesh_application_rpl_clr())
            mesh_application_factory_reset();
        break;

    case WICED_BT_MESH_CORE_STATE_TYPE_SEQ:
        // save own SEQ or RPL entry. On error do factory reset
        if (!mesh_application_seq_save(&p_state->seq))
        {
            mesh_application_factory_reset();
            break;
        }

        // only provisioner client needs to send these events to MCU
        if (!mesh_config_client)
            break;

        //WICED_BT_TRACE("mesh_state_changed_cb: addr:%x seq:%x previous_iv_idx:%d\n", p_state->seq.addr, p_state->seq.seq, p_state->seq.previous_iv_idx);
#ifdef HCI_CONTROL
        mesh_app_hci_send_seq_changed(&p_state->seq);
#endif
        break;

    case WICED_BT_MESH_CORE_STATE_NODE_STATE:
        WICED_BT_TRACE("mesh_state_changed_cb: authenticated:%d provisioned:%d proxy_on:%d pb_adv:%d\n", node_authenticated, p_state->node_state.provisioned, p_state->node_state.proxy_on, p_state->node_state.pb_adv);
        pb_gatt_in_progress = p_state->node_state.pb_adv;
        // If node is LPN in unprovisioned state(!node_authenticated) then restart scan on PB-ADV start or on provisioning end to switch scan from low duty to high duty
        if ((mesh_config.features & WICED_BT_MESH_CORE_FEATURE_BIT_LOW_POWER) != 0
            && !node_authenticated
            && ((!p_state->node_state.provisioned && p_state->node_state.pb_adv) || p_state->node_state.provisioned))
        {
            mesh_start_stop_scan_callback(WICED_FALSE, WICED_FALSE);
            mesh_start_stop_scan_callback(WICED_TRUE, WICED_FALSE);
        }
        node_authenticated = p_state->node_state.provisioned;
#ifndef MESH_HOMEKIT_COMBO_APP
        mesh_app_gatt_db_init(node_authenticated);
#else // MESH_HOMEKIT_COMBO_APP
        mesh_provisioning_state_changed(p_state->node_state.provisioned);
#endif
        if (wiced_bt_mesh_app_func_table.p_mesh_app_init != NULL)
        {
            wiced_bt_mesh_app_func_table.p_mesh_app_init(node_authenticated);
        }
        break;

    case WICED_BT_MESH_CORE_STATE_LPN_FRIENDSHIP:
        WICED_BT_TRACE("mesh_state_changed_cb:LPN_FRIENDSHIP: established:%d addr:%x/%x\n", p_state->lpn.established, p_state->lpn.addr, wiced_bt_mesh_core_lpn_get_friend_addr());
        break;

    case WICED_BT_MESH_CORE_STATE_LPN_SCAN:
        // it can happen only on LPN with established friendship
        WICED_BT_TRACE("mesh_state_changed_cb:LPN_SCAN: scan:%d\n", p_state->lpn_scan);
#ifndef PTS
        mesh_start_stop_scan_callback(p_state->lpn_scan, WICED_FALSE);
#endif
        break;

    case WICED_BT_MESH_CORE_STATE_LPN_SLEEP:
        WICED_BT_TRACE("mesh_state_changed_cb:LPN_SLEEP: timeout:%d\n", p_state->lpn_sleep);
#ifdef HCI_CONTROL
        mesh_app_hci_sleep();
#endif
        if (wiced_bt_mesh_app_func_table.p_mesh_app_lpn_sleep)
        {
            wiced_bt_mesh_app_func_table.p_mesh_app_lpn_sleep(p_state->lpn_sleep);
        }
        break;

    case WICED_BT_MESH_CORE_STATE_FRND_FRIENDSHIP:
        WICED_BT_TRACE("mesh_state_changed_cb:FRND_FRIENDSHIP: established:%d addr:%x\n", p_state->frnd.established, p_state->frnd.addr);
        break;

    default:
    	break;
    }
}

/*
 * The function goes through all models of all elements and returns number of elements on which specified model is present
 */
uint8_t number_of_elements_with_model(uint16_t model_id)
{
    uint8_t element_idx;
    uint8_t model_idx;
    uint8_t num_elements_with_model = 0;

    for (element_idx = 0; element_idx < mesh_config.elements_num; element_idx++)
    {
        for (model_idx = 0; model_idx < mesh_config.elements[element_idx].models_num; model_idx++)
        {
            if ((mesh_config.elements[element_idx].models[model_idx].company_id == MESH_COMPANY_ID_BT_SIG) &&
                (mesh_config.elements[element_idx].models[model_idx].model_id == model_id))
            {
                num_elements_with_model++;
                break;
            }
        }
    }
    return (num_elements_with_model);
}

void mesh_setup_nvram_ids()
{
    uint16_t cfg_data_len = wiced_bt_mesh_get_node_config_size(&mesh_config);

    wiced_bt_mesh_light_lc_nvram_id_start           = WICED_NVRAM_VSID_END                          - number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_LIGHT_LC_SRV);
    wiced_bt_mesh_light_hsl_nvram_id_start          = wiced_bt_mesh_light_lc_nvram_id_start         - number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_LIGHT_HSL_SRV);
    wiced_bt_mesh_light_ctl_nvram_id_start          = wiced_bt_mesh_light_hsl_nvram_id_start        - number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_LIGHT_CTL_SRV);
    wiced_bt_mesh_light_xyl_nvram_id_start          = wiced_bt_mesh_light_ctl_nvram_id_start        - number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_LIGHT_XYL_SRV);
    wiced_bt_mesh_light_lightness_nvram_id_start    = wiced_bt_mesh_light_xyl_nvram_id_start        - number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_LIGHT_LIGHTNESS_SRV);
    wiced_bt_mesh_power_level_nvram_id_start        = wiced_bt_mesh_light_lightness_nvram_id_start  - number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_GENERIC_POWER_LEVEL_SRV);
    wiced_bt_mesh_power_onoff_nvram_id_start        = wiced_bt_mesh_power_level_nvram_id_start      - number_of_elements_with_model(WICED_BT_MESH_CORE_MODEL_ID_GENERIC_POWER_ONOFF_SRV);
    wiced_bt_mesh_default_trans_time_nvram_id_start = wiced_bt_mesh_power_onoff_nvram_id_start      - 1;
    wiced_bt_mesh_scheduler_nvram_id_start          = wiced_bt_mesh_default_trans_time_nvram_id_start - wiced_bt_mesh_scheduler_events_max_num;
    wiced_bt_mesh_scene_nvram_id_end                = wiced_bt_mesh_scheduler_nvram_id_start        - 1;
    wiced_bt_mesh_scene_nvram_id_start              = wiced_bt_mesh_scene_nvram_id_end              - wiced_bt_mesh_scene_max_num;         
    wiced_bt_mesh_scene_register_nvram_id           = wiced_bt_mesh_scene_nvram_id_start            - 1;        

    wiced_bt_mesh_core_nvm_idx_node_data            = wiced_bt_mesh_scene_register_nvram_id     - 1;
    wiced_bt_mesh_core_nvm_idx_virt_addr            = wiced_bt_mesh_core_nvm_idx_node_data      - 1;
    wiced_bt_mesh_core_nvm_idx_frnd_state           = wiced_bt_mesh_core_nvm_idx_virt_addr      - 1;
    wiced_bt_mesh_core_nvm_idx_net_key_begin        = wiced_bt_mesh_core_nvm_idx_frnd_state     - wiced_bt_mesh_core_net_key_max_num;
    wiced_bt_mesh_core_nvm_idx_app_key_begin        = wiced_bt_mesh_core_nvm_idx_net_key_begin  - wiced_bt_mesh_core_app_key_max_num;
    wiced_bt_mesh_core_nvm_idx_health_state         = wiced_bt_mesh_core_nvm_idx_app_key_begin  - 1;
    wiced_bt_mesh_core_nvm_idx_cfg_data             = wiced_bt_mesh_core_nvm_idx_health_state   - ((cfg_data_len + 0xfe) / 0xff);
    mesh_nvm_idx_seq                                = wiced_bt_mesh_core_nvm_idx_cfg_data       - 1;

    WICED_BT_TRACE("setup nvram ids: net_key_max_num:%d app_key_max_num:%d nvm_idx_seq:%x %x-%x\n", wiced_bt_mesh_core_net_key_max_num, wiced_bt_mesh_core_app_key_max_num, mesh_nvm_idx_seq, wiced_bt_mesh_core_nvm_idx_cfg_data, WICED_NVRAM_VSID_END);
}

wiced_bool_t mesh_application_seq_init(void)
{
    wiced_result_t      result;
#ifdef CYW20706A2
    uint32_t            len;
#else
    mesh_app_rpl_item_t rpl_item;
#endif
    uint8_t             *p;
    uint32_t            i, seq;


    //WICED_BT_TRACE("mesh_application_seq_init:node_authenticated:%d\n", node_authenticated);

    // if node is provisioned then read own SEQ and RPL size from NVRAM, update SEQ, save it in the NVRAM and set it to the node
    // otherwise just save 0-initialized value in NVRAM
    if (node_authenticated)
    {
        if (sizeof(mesh_app_rpl_init) != mesh_nvram_access(WICED_FALSE, mesh_nvm_idx_seq, (uint8_t*)&mesh_app_rpl_init, sizeof(mesh_app_rpl_init), &result) || result != 0)
            return WICED_FALSE;
        // add 1000 because we write only multiples of thousands values. It is not needed if node doesn't send messages often.
        mesh_app_rpl_init.seq += 1000;
    }
    if (sizeof(mesh_app_rpl_init) != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq, (uint8_t*)&mesh_app_rpl_init, sizeof(mesh_app_rpl_init), &result) || result != 0)
        return WICED_FALSE;
    if (!wiced_bt_mesh_core_set_seq(0, mesh_app_rpl_init.seq, WICED_FALSE))
        return WICED_FALSE;

    // read RPL items from NVRAM and set them to the node
#ifdef CYW20706A2
    // On 20706 RPL size is hardcoded.
    // Make sure size is correct
    if(mesh_app_rpl_init.size > mesh_config.replay_cache_size)
        return WICED_FALSE;
    //Allocate memory for full RPL and read RPL entries from the NVRAM
    mesh_app_rpl_items = (mesh_app_rpl_item_t*)wiced_memory_permanent_allocate(mesh_config.replay_cache_size * sizeof(mesh_app_rpl_item_t));
    // Read RPL if it isn't empty
    if (mesh_app_rpl_init.size != 0)
    {
        len = mesh_app_rpl_init.size * sizeof(mesh_app_rpl_item_t);
        if (len != mesh_nvram_access(WICED_FALSE, mesh_nvm_idx_seq - 1, (uint8_t*)mesh_app_rpl_items, len, &result) || result != 0)
            return WICED_FALSE;
    }
    // Set RPL into the core lib
    for (i = 0; i < mesh_app_rpl_init.size; i++)
    {
        p = mesh_app_rpl_items[i].seq;
        STREAM_TO_UINT24(seq, p);
        if (!wiced_bt_mesh_core_set_seq(mesh_app_rpl_items[i].addr & (~MESH_APP_RPL_ITEM_PREV_IVI_FLAG), seq, (mesh_app_rpl_items[i].addr & MESH_APP_RPL_ITEM_PREV_IVI_FLAG) != 0 ? WICED_TRUE : WICED_FALSE))
            return WICED_FALSE;
    }
#else
    for (i = 0; i < mesh_app_rpl_init.size; i++)
    {
        if (sizeof(rpl_item) != mesh_nvram_access(WICED_FALSE, mesh_nvm_idx_seq - 1 - i, (uint8_t*)&rpl_item, sizeof(rpl_item), &result) || result != 0)
            continue;
        p = rpl_item.seq;
        STREAM_TO_UINT24(seq, p);
        if (!wiced_bt_mesh_core_set_seq(rpl_item.addr & (~MESH_APP_RPL_ITEM_PREV_IVI_FLAG), seq, (rpl_item.addr & MESH_APP_RPL_ITEM_PREV_IVI_FLAG) != 0 ? WICED_TRUE : WICED_FALSE))
            return WICED_FALSE;
    }
#endif
    return WICED_TRUE;
}

// saves own SEQ or RPL entry
wiced_bool_t mesh_application_rpl_clr(void)
{
    wiced_result_t          result;
    WICED_BT_TRACE("Clear RPL: size:%d\n", mesh_app_rpl_init.size);
    mesh_app_rpl_init.size = 0;
    if (sizeof(mesh_app_rpl_init) != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq, (uint8_t*)&mesh_app_rpl_init, sizeof(mesh_app_rpl_init), &result) || result != 0)
        return WICED_FALSE;
    return WICED_TRUE;
}

// saves own SEQ or RPL entry
wiced_bool_t mesh_application_seq_save(wiced_bt_mesh_core_state_seq_t *p_seq)
{
    wiced_result_t          result;

    // WICED_BT_TRACE("Save SRC:%x SEQ:%d prev_ivi:%d idx:%d rpl_size:%d\n", p_seq->addr, p_seq->seq, p_seq->previous_iv_idx, p_seq->rpl_entry_idx, mesh_app_rpl_init.size);

    // if it is own SEQ then save it into NVRAM
    if (p_seq->addr == 0)
    {
        // write only multiples of thousands values to prevent too often NVRAM writes. Write each change if node doesn't send messages often.
        mesh_app_rpl_init.seq = p_seq->seq;
        if ((mesh_app_rpl_init.seq % 1000) == 0)
        {
            if (sizeof(mesh_app_rpl_init) != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq, (uint8_t*)&mesh_app_rpl_init, sizeof(mesh_app_rpl_init), &result) || result != 0)
                return WICED_FALSE;
        }
    }
    else if (p_seq->addr == 0xffff)
    {
        // 0xffff means delete all RPL entries - all other fields are ignored in that case.
        mesh_app_rpl_init.size = 0;
        if (sizeof(mesh_app_rpl_init) != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq, (uint8_t*)&mesh_app_rpl_init, sizeof(mesh_app_rpl_init), &result) || result != 0)
            return WICED_FALSE;
    }
    else
    {
        // save RPL entry
        uint8_t                 *p;
#ifdef CYW20706A2
        uint32_t                len;
        mesh_app_rpl_item_t     *p_rpl_item;

        // On 20706 RPL size is hardcoded. Make sure index is inside that size
        if (p_seq->rpl_entry_idx >= mesh_config.replay_cache_size)
            return WICED_FALSE;

        // If new record has to be added
        if (p_seq->rpl_entry_idx >= mesh_app_rpl_init.size)
        {
            // delete (set invalid group address 0x8000) probably existing records for unknown indices. It is possible because we save RPL entries on successfull message handling
            while (mesh_app_rpl_init.size < p_seq->rpl_entry_idx)
                mesh_app_rpl_items[mesh_app_rpl_init.size++].addr = 0x8000;
            mesh_app_rpl_init.size = p_seq->rpl_entry_idx + 1;
            // update RPL_INIT NVRAM data with new RPL size
            if (sizeof(mesh_app_rpl_init) != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq, (uint8_t*)&mesh_app_rpl_init, sizeof(mesh_app_rpl_init), &result) || result != 0)
                return WICED_FALSE;
        }

        p_rpl_item = &mesh_app_rpl_items[p_seq->rpl_entry_idx];
        p_rpl_item->addr = p_seq->addr;
        if (p_seq->previous_iv_idx)
            p_rpl_item->addr |= MESH_APP_RPL_ITEM_PREV_IVI_FLAG;
        p = p_rpl_item->seq;
        UINT24_TO_STREAM(p, p_seq->seq);
        len = mesh_app_rpl_init.size * sizeof(mesh_app_rpl_item_t);
        if (len != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq - 1, (uint8_t*)mesh_app_rpl_items, len, &result) || result != 0)
            return WICED_FALSE;
#else
        mesh_app_rpl_item_t     rpl_item;
        rpl_item.addr = p_seq->addr;
        if (p_seq->previous_iv_idx)
            rpl_item.addr |= MESH_APP_RPL_ITEM_PREV_IVI_FLAG;
        p = rpl_item.seq;
        UINT24_TO_STREAM(p, p_seq->seq);
        if (sizeof(rpl_item) != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq - 1 - p_seq->rpl_entry_idx, (uint8_t*)&rpl_item, sizeof(rpl_item), &result) || result != 0)
            return WICED_FALSE;
        // If new record is added
        if (p_seq->rpl_entry_idx >= mesh_app_rpl_init.size)
        {
            // delete probably existing records for unknown indices. It is possible because we save RPL entries on successfull message handling
            while (mesh_app_rpl_init.size < p_seq->rpl_entry_idx)
                mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq - 1 - mesh_app_rpl_init.size++, NULL, 0, &result);
            // update RPL_INIT NVRAM data with new RPL size
            if (sizeof(mesh_app_rpl_init) != mesh_nvram_access(WICED_TRUE, mesh_nvm_idx_seq, (uint8_t*)&mesh_app_rpl_init, sizeof(mesh_app_rpl_init), &result) || result != 0)
                return WICED_FALSE;
        }
#endif
    }
    return WICED_TRUE;
}
